///////////////////////////////////
//        dashBy.js
//
// Library for real-time web dashboards
//
// (c) 2016 Sensotronica Ltd.
// Distributed under the terms of MIT License
///////////////////////////////////
/*
 declare var dashBy_options:{...} object on the webpage to override default settings below
*/

var dashBy_config = {
    'prefix': "{::",
    'postfix': "::}",
    'port': 'auto',
    'timequant': 25,
    'ws_timeout': 2000, // Websocket tiemout in msec
    'time_profiling': false,
    'simulate_websocket': false,
    'markup': 'none',
    /* available options: none, markdown, textile */
    'include_path': 'scripts',
    //    'loadCallback': function callMeWhenReady() {        alert('Iitialisation complete');    }
    //    'sendCallback': function callMeWhenSending(name,value) { console.log('Sending: '+name+'='+value);    }

};

var dashBy_channel = {};
var dashBy_control = {};
var dashBy_scripts = {};
var dashBy_includes = {};
var dashBy_count = 0;
var dashBy_lock = 0;
var dashBy_timout = 0;
var dashBy_connection = 0;
var dashBy_loadSemaphore = 0;
var dashBy_template = [];
var dashBy_preload = 0;
var dashBy_timerId = -1;

function dashBy_InitConnection() {
    dashBy_config['state'] = 'connecting';
    console.log("Connecting websocket " + "ws://" + window.location.hostname + ":" + dashBy_config.port);
    dashBy_connection = new WebSocket("ws://" + window.location.hostname + ":" + dashBy_config.port)
    dashBy_connection.onopen = function () {
        if (dashBy_connection.readyState != dashBy_connection.OPEN) return; // invalid callback
        dashBy_config['state'] = 'open';
        dashBy_timout = 0;
        console.log("Connection opened");
        for (channel in dashBy_channel) {
            dashBy_connection.send(channel + '?');
        }
    }

    dashBy_connection.onclose = function () {
        dashBy_config['state'] = 'closed';
        console.log("Connection closed");
        dashBy_connection = 0;
        setTimeout(function () {
            dashBy_InitConnection();
        }, 1000);
    }
    dashBy_connection.onerror = function () {
        dashBy_config['state'] = 'error';
        console.error("Connection error")
    }
    dashBy_connection.onmessage = function (event) {
        if (event.data.indexOf('=') >= 0) {
            var channel = event.data.split('=');
            var value = [];
            if (channel[1].indexOf('@') >= 0) {
                value = channel[1].split('@');
            } else {
                var d = new Date();
                value[0] = channel[1];
                value[1] = d.getTime();
            }
            dashBy_update(channel[0], value[0], Number(value[1]));
            channel = 0;
            value = 0;
            d = null;
        } else {
            if (event.data == '*') { // Pong recieved
                dashBy_timout = 0;
            }
        }
        event = null;
    }

    if (dashBy_timerId != -1) window.clearInterval(dashBy_timerId);

    dashBy_timerId = setInterval(function () {
        if (dashBy_config['state'] == 'open') {
            if (dashBy_connection.readyState == dashBy_connection.OPEN) {
                dashBy_connection.send('*');
            }
            dashBy_timout++;
            if ((dashBy_timout * 100) >= dashBy_config['ws_timeout']) {
                if (dashBy_connection.readyState == dashBy_connection.OPEN) {
                    dashBy_connection.close();
                    console.log('Connection timeout');
                }
            }

        };
    }, 100); // Start ping service
}


function dashBy_send(channel, value) {
    if (dashBy_config.sendCallback != undefined) {
        if (dashBy_config.sendCallback(channel, value) == false) return; // Skip sending if call back returned false
    }
    if (dashBy_connection.readyState != dashBy_connection.OPEN) return; // invalid call
    if (dashBy_config['state'] == 'open') {
        dashBy_connection.send(channel + '=' + value);
    }
}

// Cash controlled resource load

var dashBY_file_cash = {
    'control_queue': [],
    'cashed_files': []
}

function dashBy_loaderCallBack(file_name) {
    console.log('loaded: ' + file_name);
    dashBy_loadSemaphore--;
    for (ctrl_type in dashBY_file_cash['control_queue']) {
        var idx = dashBY_file_cash['control_queue'][ctrl_type].indexOf(file_name); // Check if control is awaiting this file
        if (idx >= 0) {
            dashBY_file_cash['control_queue'][ctrl_type].splice(idx, 1); // Remove file from waiting list
            if (dashBY_file_cash['control_queue'][ctrl_type].length == 0) // All required files are loaded
            {
                dashBy_InitControls(ctrl_type); // Now initialise all controls of this type
            }
        }
    }
    dashBy_loadFiles(); // Load next file if available
}

function dashBy_loadFiles() {
    for (ctrl_type in dashBY_file_cash['control_queue']) { // Scan through all controls
        if (dashBY_file_cash['control_queue'][ctrl_type].length == 0) continue;
        var file_list = dashBY_file_cash['control_queue'][ctrl_type];
        if (file_list == undefined) continue;
        if (file_list.length == 0) continue;
        for (file_index in file_list) {
            var file_name = file_list[file_index];
            if (dashBY_file_cash['cashed_files'][file_name] != undefined) {
                continue; // Skip loading cashed files
            }
            dashBY_file_cash['cashed_files'][file_name] = true; // Mark file will be cashed
            var ext = file_name.substring(file_name.lastIndexOf(".") + 1);
            if (ext == 'js') {
                dashBy_loadSemaphore++;
                dashBy_loadscript(file_name, function (name) {
                    dashBy_loaderCallBack(name);
                }, file_name)
            }
            if (ext == 'css') {
                dashBy_loadSemaphore++;
                dashBy_loadcss(file_name, function (name) {
                    dashBy_loaderCallBack(name);
                }, file_name)
            }
        }
    }
    if (dashBy_loadSemaphore <= 0) { // All control resources  loaded
        if (dashBy_config.simulate_websocket == false)  // Initalise WebSocket connection
        {
         if(dashBy_config.port=='auto')
         {
          dashBy_config.port=window.location.port;
          if(dashBy_config.port=='')dashBy_config.port='80';
         }
         console.log('WS_PORT='+dashBy_config.port);
         dashBy_InitConnection();
        }
        if (dashBy_config.loadCallback != undefined) dashBy_config.loadCallback();
    }
}

function dashBy_addLoadQueue(ctrl_type, file_names) {
    if (file_names == undefined) {
        file_names = [];
    }
    dashBY_file_cash['control_queue'][ctrl_type] = [];
    dashBY_file_cash['control_queue'][ctrl_type] = file_names;
}

function dashBy_AddCtrlResources(name) {
    var ctrlPreloaderName = 'dashBy_' + name + '_require';
    if (window[ctrlPreloaderName]) { // Control requires resources to be preloaded
        //console.log(ctrlPreloaderName);
        //console.log(window[ctrlPreloaderName]);
        dashBy_addLoadQueue(name, window[ctrlPreloaderName]);
    } else {
        dashBy_InitControls(name); // Initialise all controls that does not require extra files to load
    }

    dashBy_loadSemaphore--;
    if (dashBy_loadSemaphore == 0) // All resources are loaded
    {
        dashBy_loadFiles();
    }
}

function dashBy_InitControls(name) {
    for (var i = 0; i < dashBy_count; i++) {
        if (dashBy_control[i].type != name) continue;
        var ctrlClassName = 'dashBy_' + name;
        //        console.log("Init control " + i + " from " + dashBy_count);
        dashBy_control[i].handler = new window[ctrlClassName]('dashBY' + (i + 1), dashBy_control[i].channel, dashBy_control[i].options);
    }
}

function dashBy_load_markup(name) {
    dashBy_preload++;
    var fileref = document.createElement('script');
    fileref.setAttribute("type", "text/javascript");
    fileref.setAttribute("src", name);
    fileref.onload = function () {
        console.log("loaded: " + fileref.getAttribute("src"));
        dashBy_preload--;
        if (dashBy_preload <= 0) dashBy_Init();
    };
    document.getElementsByTagName("head")[0].appendChild(fileref)
}


function dashBy_loadcss(css_name, callback, opt) {
    var fileref = document.createElement("link")
    fileref.setAttribute("rel", "stylesheet")
    fileref.setAttribute("type", "text/css")
    fileref.setAttribute("href", css_name)
    fileref.onload = function () {
        if (callback !== undefined) callback(opt);
    };
    document.getElementsByTagName("head")[0].appendChild(fileref)
}

function dashBy_loadscript(script_name, callback, opt) {
    var fileref = document.createElement('script');
    fileref.setAttribute("type", "text/javascript");
    fileref.setAttribute("src", script_name);
    fileref.onload = function () {
        if (callback !== undefined) callback(opt);
    };
    document.getElementsByTagName("head")[0].appendChild(fileref)
}

function dashBy_LoadControls() {
    for (var i = 0; i < dashBy_count; i++) // collect control types
    {
        dashBy_scripts[dashBy_control[i].type] = 0;
    }

    for (var key in dashBy_scripts) // Load control scripts
    {
        dashBy_loadscript('controls/dashBy_' + key + '.js', dashBy_AddCtrlResources, key);
        dashBy_loadSemaphore++;
    }
}

function dashBy_processTemplate(a) // Process control template
{
    var p1 = a.indexOf('('); //
    if (p1 <= 0) return "dashBy: Template Error, expect ( ";
    var p2 = a.indexOf(')'); // extract template name
    if (p2 < p1) return "dashBy: Template Error, expect ) ";
    dashBy_template[a.substring(p1 + 1, p2).trim()] = a; // Store template
    return '';
}

function dashBy_makeControlFromTempate(template, varlist) // Process control definition
{
    if (dashBy_template[template] == undefined) {
        console.log('dashBy: template <' + template + '> is not defined');
        return '<span>' + template + ' ???</span>';
    }
    var a = dashBy_template[template];
    var p1 = a.indexOf('('); // get control type
    if (p1 <= 0) return "dashBy: Error, expect ( ";
    var p2 = a.indexOf(')'); // extract variables list
    if (p2 < p1) return "dashBy: Error, expect ) ";
    return dashBy_processControl(a.substring(0, p1 + 1) + varlist + a.substring(p2));
}

function dashBy_processControl(a) // Process control definition
{
    if (a == '|') return '<!-- .Purifier-col -->';
    if (a == '/') return '<!-- .Purifier-raw -->';
    if (a == '+') return '<!-- .Purifier-nested-grid-begin -->';
    if (a == '-') return '<!-- .Purifier-nested-grid-end -->';

    if (a.indexOf('?') >= 0) { // Template
        dashBy_processTemplate(a.substring(1));
        return '';
    }

    if (a.indexOf('$') >= 0) { // Template
        console.log('Error: unprocessed include: ' + a.substring(1));
        return 'Error: Inc(' + a.substring(1) + ')';
    }

    var br = a.indexOf('{')
    if ((br >= 0)&&(br < 4))  { // Absolute positioning
       var p1 = a.indexOf('{'); // get position params
       var p2 = a.indexOf('}'); 
       if (p2 < p1) return "dashBy: Error, expect } ";
       var htmlParams = a.substring(p1 + 1, p2);
       var p3 = htmlParams.indexOf(',');
       if (p3 <= 0) return "dashBy: Error, expect { x, y} in  " + a;
       var posX = htmlParams.substring(0,p3);
       var posY = htmlParams.substring(p3+1);
        return "<div style='position:absolute; left:"+posX+"px; top:"+posY+"px;' >\n"+dashBy_processControl(a.substring(p2+1))+"\n</div>";
    }
    


    var p1 = a.indexOf('('); // get control type
    if (p1 <= 0) return "dashBy: Error, expect ( .. " + a;
    var p2 = a.indexOf(')'); // extract variables list
    if (p2 < p1) return "dashBy: Error, expect ) ";

    if (dashBy_template[a.substring(0, p1).trim()] != undefined) {
        return dashBy_makeControlFromTempate(a.substring(0, p1), a.substring(p1 + 1, p2));
    }

    // prepare variables list
    var medVar = a.substring(p1 + 1, p2).split(',');
    var varList = [];
    for (var i = 0; i < medVar.length; i++) {
        medVar[i] = medVar[i].trim();
        var b1 = medVar[i].indexOf('['); // Check for variable range
        if (b1 <= 0) { // no range defined
            varList.push(medVar[i]); // Append variable to the list
        } else { // Variable range defined
            var b2 = medVar[i].indexOf(']'); // Read range first & last value
            if (b2 <= 0) return "dashBy: Error, expect ] ";
            var dd = medVar[i].indexOf(':');
            if (dd <= 0) return "dashBy: Error, expect : ";
            var first = Math.round(medVar[i].substring(b1 + 1, dd));
            var last = Math.round(medVar[i].substring(dd + 1, b2));
            if (Math.abs(last - first) > 128) return "dashBy: too big range for the variables ";
            var prefix = medVar[i].substring(0, b1);
            var postfix = medVar[i].substring(b2 + 1);
            var step = 1;
            if (last < first) step = -1; // Loo for ascending or descending steps
            varList.push(prefix + first + postfix); // Add first var
            do {
                first += step;
                varList.push(prefix + first + postfix); // Add other vars in the list
            } while (first != last);
        }
    }

    // extract parameters
    var parlist = a.substring(p2 + 1).split('}');
    for (var i = 0; i < parlist.length; i++) // trim all parameters
    {
        var p = parlist[i].indexOf('{');
        if (p >= 0) parlist[i] = parlist[i].substring(p + 1);
        parlist[i] = parlist[i].split(',');
        for (var j = 0; j < parlist[i].length; j++) parlist[i][j] = parlist[i][j].trim();
        if (parlist.length > 0) {
            if (parlist[parlist.length - 1].length == 0) {
                parlist.pop();
            }
        }
    }

    var control = {
        'type': a.substring(0, p1),
        'channel': varList,
        'options': parlist,
        'handler': null,
        'timing': 0,
        'pending': 0
    };
    dashBy_control[dashBy_count++] = control;

    return "<span class='dashBy' id='dashBY" + dashBy_count + "'>Loading(" + dashBy_count + ")...</span>";
}

function dashBy_update(channel, value, ts) {
    if ((dashBy_channel[channel] == null) || (dashBy_channel[channel].length === 0)) return;
    for (var i = 0; i < dashBy_channel[channel].length; i++) {
        if (dashBy_control[dashBy_channel[channel][i]].handler !== null) {
            dashBy_control[dashBy_channel[channel][i]].handler.update(channel, value, ts);
        }
    }
}

function dashBy_redraw() {
    for (var i = 0; i < dashBy_count; i++) {
        if (dashBy_control[i].handler !== null) {
            dashBy_control[i].pending += dashBy_config.timequant;
            if ((dashBy_control[i].pending >= (dashBy_control[i].timing * 2)) || (dashBy_control[i].pending >= 5000)) {
                dashBy_control[i].pending = 0;
                var b = new Date();
                dashBy_control[i].handler.redraw();
                var e = new Date();
                dashBy_control[i].timing = e.getTime() - b.getTime();
                if (dashBy_config.time_profiling) {
                    dashBy_send('dashBy.time_profiling.' + dashBy_control[i].type + '.control' + i + '.redrawtime', dashBy_control[i].timing);
                }
            }
        }
    }
}

(function render() {
    dashBy_redraw();
    setTimeout(render, dashBy_config.timequant);
})();


//  Preprocessor: include server side files

function parseINC() {
    var obj = this;
    var cached = false;

    if (arguments.length > 1) {
        obj = arguments[0];
        cached = true;
    }


    if (obj.status == 200) // HTTP OK code
    {
        var str = obj.responseText;
        if (cached != true) { // Add to include file cache
            dashBy_includes[obj.filename] = obj.responseText;
        }
        if (obj.options != undefined) {
            var opt_list = obj.options.split(',');
            for (var i = 0; i < opt_list.length; i++) {
                var pair = opt_list[i].split('=');
                if (pair.length >= 2) {
                    str = str.split(pair[0]).join(pair[1]);
                }
            }
        }
        dashBy_Include(str);
        if (cached == true) {
            console.log('got cached: ' + obj.responseURL);
        } else {
            console.log('loaded: ' + obj.responseURL);
        }
    } else {
        dashBy_Include('ErrorLoad: ' + obj.responseURL + ' : ' + obj.responseText + ' ');
        console.log('ErrorLoad: ' + obj.responseURL + ' : ' + obj.responseText + ' ');
    }
}

function loadINC(file) {
    var request = new XMLHttpRequest();

    var p = file.indexOf('(');
    if (p >= 0) {
        var e = file.indexOf(')');
        if (e <= p) {
            console.log('Incude Warning: missed )');
            e = p.strlen() - 1;
        }
        request.options = file.substring(p + 1, e);
        request.filename = file.substring(0, p) + '.dby';
    } else {
        request.filename = file + '.dby';
    }
    if (dashBy_config['include_path'].length > 0) {
        request.filename = dashBy_config['include_path'] + '/' + request.filename;
    }
    if (dashBy_includes[request.filename] != undefined) { // File is chached?
        var obj = {
            status: 200,
            filename: request.filename,
            responseURL: request.filename,
            responseText: dashBy_includes[request.filename],
            options: request.options
        };
        parseINC(obj, true);
    } else {
        request.addEventListener("load", parseINC);
        request.open('GET', request.filename);
        request.send();
    }
}

function dashBy_Include(contents) {
    var elems = document.body.getElementsByTagName("*");
    var length = elems.length;

    if (contents == undefined) { // Search  for include tags
        for (var i = 0; i < length; ++i) {
            var item = elems[i];
            str = item.innerHTML;
            var p1 = str.indexOf(dashBy_config.prefix + '$');
            if (p1 < 0) continue;
            var p2 = str.substring(p1).indexOf(dashBy_config.postfix);
            if (p2 < 0) continue;
            p2 += p1;
            loadINC(str.substring(p1 + dashBy_config.prefix.length + 1, p2).trim());
            //        console.log('Requesting: ' + str.substring(p1 + dashBy_config.prefix.length + 1, p2).trim());
            return;
        }
        dashBy_ProcessPage(); // If we come here then no more include tags left
    } else { // Insert retrived contents
        for (var i = 0; i < length; ++i) {
            var item = elems[i];
            str = item.innerHTML;
            var p1 = str.indexOf(dashBy_config.prefix + '$');
            if (p1 < 0) continue;
            var p2 = str.substring(p1).indexOf(dashBy_config.postfix);
            if (p2 < 0) continue;
            p2 += p1;
            item.innerHTML = str.substring(0, p1) + contents + str.substring(p2 + dashBy_config.postfix.length);
            return dashBy_Include();
        }
    }
}


function dashBy_Init() {
    dashBy_Include();
}

function verifyStyle(selector) {
    var rules;
    var haveRule = false;

    if (typeof document.styleSheets != "undefined") { //is this supported
        var cssSheets = document.styleSheets;

        outerloop:
            for (var i = 0; i < cssSheets.length; i++) {

                //using IE or FireFox/Standards Compliant
                rules = (typeof cssSheets[i].cssRules != "undefined") ? cssSheets[i].cssRules : cssSheets[i].rules;

                for (var j = 0; j < rules.length; j++) {

                    if (rules[j].selectorText == selector) {
                        haveRule = true;
                        break outerloop;
                    }
                } //innerloop

            } //outer loop
    } //endif

    return haveRule;
} //

function dashBy_ProcessPage() // Parse HTML body for dashBy tags
{
    // First pass - make layour for purecss grid
    var elems = document.body.getElementsByTagName("*");
    var length = elems.length;

    if (verifyStyle('.pure-g')) { // Check if dashBy.css is loaded
        for (var i = 0; i < length; ++i) {
            var item = elems[i];
            str = dashBy_purifier(item.innerHTML);
            item.innerHTML = str;
        }
    } else console.log('info: no css included');

    // Second pass - apply markup laguage conversion when necessary
    if ((dashBy_config['markup'] == 'markdown') || (dashBy_config['markup'] == 'textile')) {
        elems = document.body.getElementsByClassName("dashBy_" + dashBy_config['markup']);
        length = elems.length;
        for (var i = 0; i < length; ++i) {
            var item = elems[i];
            var str = item.innerHTML;
            if (dashBy_config['markup'] == 'markdown') {
                str = marked(str);
            } else {
                if (dashBy_config['markup'] == 'textile') {
                    str = textile(str);
                }
            }
            item.innerHTML = str;
        }
    }

    // Third pass - apply controls conversion
    elems = document.body.getElementsByTagName("*");

    length = elems.length;

    for (var i = 0; i < length; ++i) {
        var item = elems[i];
        var str = item.innerHTML;
        for (var ctr = 0; ctr < 8000; ctr++) { /*Not more then 1000 controls */
            var p1 = str.indexOf(dashBy_config.prefix);
            if (p1 < 0) break;
            var p2 = str.indexOf(dashBy_config.postfix);
            if (p1 < 0) break;
            if (p1 >= 0 && p2 > p1) {
                str = str.substring(0, p1) + dashBy_processControl(str.substring(p1 + dashBy_config.prefix.length, p2).trim()) + str.substring(p2 + dashBy_config.postfix.length);
            } else break;
        }
        item.innerHTML = str;
    }


    // Compose variables dispatch table
    for (var i = 0; i < dashBy_count; i++) {
        if (Array.isArray(dashBy_control[i].channel)) {
            for (var v = 0; v < dashBy_control[i].channel.length; v++) {
                if (dashBy_channel.hasOwnProperty(dashBy_control[i].channel[v]) === false) { // Fist  occurence
                    dashBy_channel[dashBy_control[i].channel[v]] = new Array();
                    dashBy_channel[dashBy_control[i].channel[v]][0] = i;
                } else {
                    dashBy_channel[dashBy_control[i].channel[v]][dashBy_channel[dashBy_control[i].channel[v]].length] = i;
                }
            }
        } else {
            if (typeof dashBy_control[i].channel === 'string') {
                if (dashBy_channel.hasOwnProperty(dashBy_control[i].channel) === false) { // Fist  occurence
                    dashBy_channel[dashBy_control[i].channel] = new Array();
                    dashBy_channel[dashBy_control[i].channel][0] = i;
                } else {
                    dashBy_channel[dashBy_control[i].channel][dashBy_channel[dashBy_control[i].channel].length] = i;
                }
            }
        }
    }

    dashBy_LoadControls();
}

(function () {
    document.addEventListener('DOMContentLoaded', function () {
        if (window['dashBy_options'] !== undefined) {
            for (var opt in dashBy_config) { // Copy  user defined options
                if (dashBy_options[opt] !== undefined) dashBy_config[opt] = dashBy_options[opt];
            }
        }

        if (dashBy_config['markup'] == 'markdown') {
            dashBy_load_markup('lib/marked.js');
        } else {
            if (dashBy_config['markup'] == 'textile') {
                dashBy_load_markup('lib/textile.js');
            } else {
                dashBy_Init();
            }
        }
    }, false);
})();

// --------- Automatic generation of purecss layout  --------

var dashBy_grid = {
    large: [
  ["1"],
  ["1-2", "1-2"],
  ["1-3", "1-3", "1-3"],
  ["2-5", "1-5", "1-5", "1-5"],
  ["1-5", "1-5", "1-5", "1-5", "1-5"]
],
    middle: [
      ["1"],
      ["1", "1"],
      ["1", "1", "1"],
      ["1-2", "1-2", "1-2", "1-2"],
      ["1-2", "1-2", "1-2", "1-2", "1"]
 ],
    small: [
      ["1"],
      ["1", "1"],
      ["1", "1", "1"],
      ["1", "1", "1", "1"],
      ["1", "1", "1", "1", "1"]
 ]
};

function dashBy_pure_split(str) {
    var res = '';
    var myClass = 'dashBy_' + dashBy_config['markup'];
    var cols = str.split(dashBy_config.prefix + '|' + dashBy_config.postfix);
    var l = cols.length;
    if (cols[l - 1].length < 2) l--; //drop trailing empty oblects
    if (l > 5) {
        console.log("Purifier: Warning - column count exceeds 5; extra col. will be dropped");
        l = 5;
    }
    for (var i = 0; i < l; i++) {
        var clname = "pure-xl-" + dashBy_grid.large[l - 1][i] + " pure-lg-" + dashBy_grid.large[l - 1][i] + " pure-ml-" + dashBy_grid.middle[l - 1][i] + " pure-sl-" + dashBy_grid.small[l - 1][i] + " l-box ";
        res += "<div class='" + myClass + " " + clname + "'>" + cols[i] + "</div>\n";
    }
    return res;
}

function dashBy_purifier(str) {
    var res = '';
    var raws = str.split(dashBy_config.prefix + '/' + dashBy_config.postfix);
    var l = raws.length;
    for (var i = 0; i < l; i++) {
        res += "<div class='pure-g'>\n" + dashBy_pure_split(raws[i]) + "</div>\n";
        //res += "<div class='pure-g'>\n" + raws[i] + "</div>\n";
    }
    return res;
}

/// Below is nothing
function dashBy_purifier(str) {
    var res = '';
    var raws = str.split(dashBy_config.prefix + '/' + dashBy_config.postfix);
    var l = raws.length;
    for (var i = 0; i < l; i++) {
        res += "<div class='pure-g'>\n" + dashBy_pure_split(raws[i]) + "</div>\n";
        //res += "<div class='pure-g'>\n" + raws[i] + "</div>\n";
    }
    return res;
}

/// Below is nothing
