///////////////////////////////////
//        dashByServer.js
//
// Server side application for dashBy.js
// Routes data between Websocket and TCP
// Provides minimalistic HTTP server
//
// (c) 2016 Sensotronica Ltd.
// Distributed under the terms of MIT License
///////////////////////////////////

var http = require("http")
var WebSocketServer = require('./websocket').server;
var net = require('net');
var url = require("url")
var path = require("path")
var fs = require("fs")
var ini = require("./ini")


var Config = {
    'http_port': 8080,
    'websock_timeout': 2000,
    'TCP_port': 8000
};

var ini_filename = '';

// -- Begin command line parameters
var cmd = '';
process.argv.forEach(function (val, index, array) {
    if (val.substring(0, 1) == '-') // Option prefix
    {
        cmd = val;
    } else {
        if (cmd == '-c') ini_filename = val; // Ini file option
    }
});
// -- End command line parameters

// -- Begin Processs ini file
if (ini_filename.length > 1) {
    var ini_config = ini.parse(fs.readFileSync(ini_filename, 'utf-8'));
    if (ini_config["http"] != undefined) {
        if (ini_config["http"]["port"] != undefined)
            Config.http_port = ini_config["http"]["port"];
    }

    if (ini_config["websock"] != undefined) {
        if (ini_config["websock"]["timeout"] != undefined)
            Config.websock_timeout = ini_config["websock"]["timeout"];
    }

    if (ini_config["TCP"] != undefined) {
        if (ini_config["TCP"]["port"] != undefined)
            Config.TCP_port = ini_config["TCP"]["port"];
    }
}
// -- End process ini files

// -- Begin Webserver part
if ((Config.http_port.toString().length > 1) && (Config.http_port.toString().trim() != 'none')) {
    console.log('Starting HTTP Webserver at: ' + Config.http_port);
    var httpsrv = http.createServer(function (request, response) {
        if (request.method == 'POST') {
            var uri = url.parse(request.url).pathname,
                filename = path.join(process.cwd(), uri);
            console.log("POST:" + filename);
            var body = '';
            request.on('data', function (data) {
                body += data;
                console.log("Partial body: " + unescape(body.replace(/\+/g, ' ')));
            });
            request.on('end', function () {
                console.log("Body: " + unescape(body.replace(/\+/g, ' ')));
            });
            response.writeHead(200, {
                'Content-Type': 'text/html'
            });
            response.end('post received');
        } else if (request.method == 'GET') {
            var uri = url.parse(request.url).pathname,
                filename = path.join(process.cwd(), uri);

            var contentTypesByExtension = {
                '.html': "text/html",
                '.css': "text/css",
                '.js': "text/javascript",
                '.dby': "text/plain"
            };

            console.log('HTTP GET: ' + filename);

            fs.exists(filename, function (exists) {
                if (!exists) {
                    console.log('HTTP 1');
                    response.writeHead(404, {
                        "Content-Type": "text/plain"
                    });
                    response.write("404 Not Found\n");
                    response.end();
                    return;
                }

                if (fs.statSync(filename).isDirectory()) filename += '/index.html';

                fs.readFile(filename, "binary", function (err, file) {
                    if (err) {

                        response.writeHead(500, {
                            "Content-Type": "text/plain"
                        });
                        response.write(err + "\n");
                        response.end();
                        return;
                    }

                    var headers = {};
                    var contentType = contentTypesByExtension[path.extname(filename)];
                    if (contentType) headers["Content-Type"] = contentType;
                    response.writeHead(200, headers);
                    response.write(file, "binary");
                    response.end();

                });
            });
        } else {
            console.log('HTTP: Unsupproted request: ' + request);
        }
    }).listen(Config.http_port);

    httpsrv.on('error', function (e) {
        console.log('HTTP server error' + e);
    });
}

// -- End webserver part

// -- Begin websocket part

if (httpsrv !== undefined) {
	console.log('Starting Websocket Server at: ' + Config.http_port);

	var wsServer = new WebSocketServer({
		httpServer: httpsrv,
		autoAcceptConnections: true
	});

	wsServer.on('connect', function(connection) {
		connection.nickname = Date.now().toString();
		console.log('Connection <' + connection.nickname + '> opened');
		connection.timout = Config.websock_timeout;

		(function watchdog() {
			if (connection.connected) {
				if (connection.timout <= 0) {
					console.log('Connection <' + connection.nickname + '> timed out');
					connection.close();
					return;
				} else {
					connection.timout -= 100;
				}
				setTimeout(watchdog, 100);
			}
		})();

		connection.on('message', function(message) {
			if (message.type === 'utf8') {
				connection.timout = Config.websock_timeout;
				var str = message.utf8Data;
				//console.log("recieved: " + str);
				if (str.indexOf('=') >= 0) {
					wsServer.broadcastUTF(str);
					TCP_broadcast('websocket', str + '\n');
				} else if (str === '*') { // Keepalive Ping
					connection.sendUTF('*'); // Send pong
				}
			}
	    });

		connection.on('close', function(reasonCode, description) {
			wsServer.broadcastUTF(connection.nickname + ' left');
			console.log('Connection <' + connection.nickname + '> closed')
		});

		connection.on('error', function (e) {
			console.log('Websock connection <' + connection.nickname + '> error', e.message);
		});

	});

	wsServer.on('error', function (e) {
		console.log('Websock server error', e);
	});
}
// -- End websocket part

//-- Begin of TCP part ---

var TCP_sockets = [];
var guestId = 0;

var TCP_server = net.createServer(function (TCP_socket) {
    // Increment
    guestId++;

    TCP_socket.nickname = "Clent" + guestId;
    var clientName = TCP_socket.nickname;

    TCP_sockets.push(TCP_socket);

    // Log it to the server output
    console.log(clientName + ' Connected.');


    // When client sends data
    TCP_socket.on('data', function (data) {
        var message = data.toString();

        TCP_broadcast(clientName, message);
        wsServer.broadcastUTF(message);

        // Log it to the server output
        //        process.stdout.write(message);
    });


    // When client leaves
    TCP_socket.on('end', function () {

        // Log it to the server output
        console.log(clientName + ' Disconnected.');

        // Remove client from socket array
        removeSocket(TCP_socket);

        // Notify all clients
        //        TCP_broadcast(clientName, message);
    });


    // When socket gets errors
    TCP_socket.on('error', function (error) {
        console.log('Socket got problems: ', error.message);
    });
});


// Broadcast to others, excluding the sender
function TCP_broadcast(from, message) {

    // If there are no sockets, then don't broadcast any messages
    if (TCP_sockets.length === 0) {
 //       process.stdout.write('No clients connected\n');
        return;
    }

    // If there are clients remaining then broadcast message
    TCP_sockets.forEach(function (TCP_socket, index, array) {
        // Dont send any messages to the sender
        if (TCP_socket.nickname === from) return;

        TCP_socket.write(message);

    });

};

// Remove disconnected client from sockets array
function removeSocket(socket) {

    TCP_sockets.splice(TCP_sockets.indexOf(socket), 1);

};


// Listening for any problems with the server
TCP_server.on('error', function (error) {

    console.log("So we got problems!", error.message);

});

// Listen for a port to telnet to
// then in the terminal just run 'telnet localhost [port]'
TCP_server.listen(Config.TCP_port, function () {

    console.log("TCP Server listening at http://localhost:" + Config.TCP_port);

});

//-- End of TCP Part ---
