var _und = require('underscore');
var util = require('util');
var EventEmitter = require('events').EventEmitter;
var net = require('net');
var Parser = require('./regexp_stream');

util.inherits(MediatorInterface, EventEmitter);

function MediatorInterface(params) {
	EventEmitter.call(this);

	this.address = {
		host: params.host,
		port: params.port
	};
	this.mask = params.mask || '*';
	this.timestamp = params.timestamp || 'none';

	this._autoReconnect = (params.autoReconnect !== false); //explicit conversion to bool type. Default: true
	this._boundEmitUpdate = EventEmitter.prototype.emit.bind(this, 'update');

	this._varCache = {};

	setImmediate(MediatorInterface.prototype._open.bind(this)); //delay connecting attempts
}

MediatorInterface.prototype.mask = null; //notify mask
MediatorInterface.prototype.timestamp = null; //desired timestamp format. should be one of the: 'none', 'abs' or 'rel'
MediatorInterface.prototype.address = null; //remote endpoint address

MediatorInterface.prototype._autoReconnect = false;
MediatorInterface.prototype._boundEmitUpdate = null;
MediatorInterface.prototype._socket = null;
MediatorInterface.prototype._varCache = null; //hash map [varpath -> EventEmitter]

MediatorInterface.prototype._open = function() {
	if (this.isOpen) return;

	this._socket = net.createConnection(this.address);
	this._socket.on('connect', MediatorInterface.prototype._onSocketConnect.bind(this));
	this._socket.on('close', MediatorInterface.prototype._onSocketClose.bind(this));
	this._socket.on('error', EventEmitter.prototype.emit.bind(this, 'error')); //delegate 'error' notification out to upper level
};
MediatorInterface.prototype._onSocketConnect = function() {
	this.writeVar({
		path: '#notify',
		value: this.mask
	});
	this.writeVar({
		path: '#timestamp',
		value: this.timestamp
	});

	var parser = new Parser(/\r?\n/);
	parser.on('data', MediatorInterface.prototype._handlePacket.bind(this));
	this._socket.pipe(parser);

	this.isOpen = true;
	this.emit('connect');
};
MediatorInterface.prototype._onSocketClose = function() {
	if (this._autoReconnect) {
		//scedule next connection attempt
		setTimeout(MediatorInterface.prototype._open.bind(this), 5000);
	}

	if (!this.isOpen) return;
	this.isOpen = false;
	this.emit('close');
};
MediatorInterface.prototype._handlePacket = function(packet) {
	//here we are expecting packet to have one of the following structures:
	//1. path=value | path@=value //without timestamp
	//2. path@1234=value //absolute timestamp
	//3. path@-1234=value //relative timestamp
	// var match = packet.match(/^(.+)$/);
	var match = /^([^@=]+)@?(-?\d+)?=(.*)$/.exec(packet);
	if (!match) return;
	

	var hasTimestamp = (match[2] !== undefined);
	var relativeTs = hasTimestamp && (match[2][0] === '-');
	var ts = hasTimestamp ? parseInt(match[2]) : _und.now();
	if (relativeTs) ts += _und.now();

	var entry = {
		path: match[1],
		value: match[3],
		ts: ts
	};

	this._getEmitterOrCreate(entry.path).emit('update', entry); //notify about update
};
MediatorInterface.prototype._getEmitterOrCreate = function(varpath) {
	var emitter = this._varCache[varpath];

	if (!emitter) { //checking if var already persists in storage
		emitter = new EventEmitter();
		emitter.on('update', this._boundEmitUpdate);

		this._varCache[varpath] = emitter; //store entry to cache
	}
	return emitter;
};

MediatorInterface.prototype.isOpen = false;
MediatorInterface.prototype.subscribeUpdate = function(varpath, handler) {
	this._getEmitterOrCreate(varpath).on('update', handler);
};
MediatorInterface.prototype.unsubscribeUpdate = function(varpath, handler) {
	this._getEmitterOrCreate(varpath).removeListener('update', handler);
};
MediatorInterface.prototype.writeVar = function(entry, callback) {
	var msg = [
		entry.path,
		entry.ts ? ('@' + entry.ts) : '',
		'=',
		entry.value,
		'\r\n'
	].join('');

	this._socket.write(msg, callback);
};
MediatorInterface.prototype.readVar = function(varpath, defaultValue) {
	var request = varpath + '?';
	if (defaultValue) {
		request += ('=' + defaultValue);
	}
	request += '\r\n';
	this._socket.write(request);
};

module.exports = MediatorInterface;